module net.BurtonRadons.dedit.projectCommands;

import net.BurtonRadons.dedit.main;

import std.outbuffer;

static char [] [] filenameSplit (char [] filename)
{
    char [] [] list;
    int c, s;
    
    for (c = s = 0; c < filename.length + 1; c ++)
        if (c == filename.length
            || filename [c] == '/'
            || filename [c] == '\\'
            || filename [c] == ':')
        {
            list ~= filename [s .. c];
            s = c;
        }
        
        return list;
}

static char [] filenameJoin (char [] [] list)
{
    char [] output;
    int length;
    
    for (int c; c < list.length; c ++)
        length += list [c].length;
    
    output.length = length;
    length = 0;
    
    for (int c; c < list.length; c ++)
    {
        output [length .. length + list [c].length] = list [c];
        length += list [c].length;
    }
    
    return output;
}

static bit filenameEquals (char [] a, char [] b)
{
    for (int c; c < a.length && c < b.length; c ++)
        if (!fncharmatch (a [c], b [c]))
            return false;
        
        return true;
}

static char [] relativeFilename (char [] filename, char [] path)
{
    char [] file;
    char [] [] fn;
    char [] [] pn;
    char [] [] on;
    
    fn = filenameSplit (filename);
    pn = filenameSplit (path);
    
    file = fn [fn.length - 1];
    fn = fn [0 .. fn.length - 1];
    
    int c;
    
    for (c = 0; c < fn.length && c < pn.length; c ++)
        if (!filenameEquals (fn [c], pn [c]))
            break;
        
        for (int d = c; d < pn.length; d ++)
            on ~= d < pn.length - 1 ? (char []) "../" : (char []) "..";
        
        if (c == pn.length)
            on ~= ".";
        
        for (int d = c; d < fn.length; d ++)
            on ~= fn [d];
        
        return filenameJoin (on) ~ file;
}

static char [] absoluteFilename (char [] filename, char [] path)
{
    char [] file;
    char [] [] fn;
    char [] [] pn;
    char [] [] on;
    
    fn = filenameSplit (filename);
    pn = filenameSplit (path);
    
    file = fn [fn.length - 1];
    fn = fn [0 .. fn.length - 1];
    
    int ps = pn.length, fs = 0;
    
    if (fn [0] == ".")
        fs ++;
    else
    {
        for (int c; c < fn.length && (fn [c] == ".." || fn [c] == "/.." || fn [c] == "\\.."); c ++)
            ps --, fs ++;
        
        if (fs == 0)
            return filenameJoin (fn) ~ file;
    }
    
    for (int c; c < ps; c ++)
        on ~= pn [c];
    
    for (int c = fs; c < fn.length; c ++)
        on ~= fn [c];
    
    return filenameJoin (on) ~ file;
}

private static void relativeFilenameTest (char [] filename, char [] path, char [] result)
{
    assert (relativeFilename (filename, path) == result);
}

private static void absoluteFilenameTest (char [] filename, char [] path, char [] result)
{
    if (absoluteFilename (filename, path) != result)
    {
        printf ("absoluteFilename ('%.*s', '%.*s') != '%.*s', but rather '%.*s'\n", filename, path, result, absoluteFilename (filename, path));
        assert (absoluteFilename (filename, path) == result);
    }
}

unittest
{
    alias relativeFilenameTest r;
    alias absoluteFilenameTest a;
    
    r ("/a/b/c", "/a/b/c/d", "../../c");
    a ("../../c", "/a/b/c/d", "/a/b/c");
    a ("./q/r", "p", "p/q/r");
}

class CreateFolder
{
    Frame frame;
    ProjectView.Folder parent;
    EditText edit;

    this (View view)
    {
        GroupBox box;
        Pane pane;

        parent = view.currentFolder ();

        with (frame = new Frame ())
        {
            caption ("Create Folder");
            maximizable (false);
            resizable (false);
            minimizable (false);
        }

        with (box = new GroupBox (frame))
        {
            caption ("Folder Name");
            grid (0, 0);
            sticky ("<>");
        }

        with (edit = new EditText (box))
        {
            grid (0, 0);
            sticky ("<>");
            width (200);
            bind ("Return", &doOkay);
            bind ("Escape", &doCancel);
            makeFocus ();
            pad (5, 8);
        }

        with (pane = new Pane (frame))
        {
            grid (0, 1);
            sticky (">");
        }

        with (new Button (pane))
        {
            caption ("Okay");
            grid (0, 0);
            bordered (true);
            width (100);
            onClick.add (&doOkay);
        }

        with (new Button (pane))
        {
            caption ("Cancel");
            grid (1, 0);
            width (100);
            onClick.add (&doCancel);
        }

        frame.display ();
    }

    void doOkay (Event e)
    {
        char [] name = edit.text ();

        if (name.length)
        {
            parent.add (new ProjectView.Folder (name));
            global.projectView.paint ();
            delete frame;
        }
        else
            frame.messageBox ("Folder Name Empty", "You need to insert a name for the folder.");
    }

    void doCancel (Event e)
    {
        delete frame;
    }
}

class RenameFolder
{
    Frame frame;
    ProjectView.Folder folder;
    EditText edit;

    this (View view, ProjectView.Folder folder)
    {
        GroupBox box;
        Pane pane;

        this.folder = folder;

        with (frame = new Frame ())
        {
            caption ("Create Folder");
            maximizable (false);
            resizable (false);
            minimizable (false);
        }

        with (box = new GroupBox (frame))
        {
            caption ("Folder Name");
            grid (0, 0);
            sticky ("<>");
        }

        with (edit = new EditText (box))
        {
            grid (0, 0);
            sticky ("<>");
            width (200);
            text (folder.name ());
        }

        with (pane = new Pane (frame))
        {
            grid (0, 1);
            sticky (">");
        }

        with (new Button (pane))
        {
            caption ("Okay");
            grid (0, 0);
            bordered (true);
            width (100);
            onClick.add (&doOkay);
        }

        with (new Button (pane))
        {
            caption ("Cancel");
            grid (1, 0);
            width (100);
            onClick.add (&doCancel);
        }

        frame.display ();
    }

    void doOkay ()
    {
        folder.name (edit.text ());
        global.projectView.paint ();
        delete frame;
    }

    void doCancel ()
    {
        delete frame;
    }
}

/** Provides the project-handling command set. */
class ProjectCommands
{
    static this ()
    {
        new ProjectCommands ();
    }

    this ()
    {
        alias View.addCommand e;

        e ("ProjectNew", "", &ProjectNew,
            "Start a new project.",
            "Start a new project.  If the current project has no name, it is discarded, "
            "otherwise it is first saved.  The new project will be empty except for "
            "a new created file.");
        e ("ProjectSave", ".", &ProjectSave,
            "Save the current project.",
            "Save the current project.  If it has no name, a dialog is shown asking "
            "the user for one.  If this dialog is canceled, the save is canceled.  "
            "Projects with names are automatically saved on program exit.");
        e ("ProjectOpen", ".", &ProjectOpen,
            "Load a project.",
            "Show a dialog to load a new project.  If the dialog is canceled, nothing "
            "happens.  If the current project is not saved, then it is discarded, otherwise "
            "it is saved before loading the other project.");
        e ("ProjectReload", "", &ProjectReload,
            "Reload the current project.",
            "Reload the current project.  If the current project is not named, then nothing "
            "will happen.");
        e ("ProjectNewFolder", ".", &ProjectNewFolder,
            "Create a new folder.",
            "Create a new project folder by showing a dialog asking the user for a name.  "
            "The folder is created in either the folder of the current document or the current "
            "folder as assigned by a right-click on a folder name in the project view and "
            "selecting a menu item there.  If the "
            "dialog is canceled, nothing happens.");
        e ("ProjectDeleteFolder", "", &ProjectDeleteFolder,
            "Delete the current folder and close content.",
            "Delete the current folder and close all documents within it, recursively.  "
            "Documents with no name are given a dialog to save their name (if this "
            "dialog is canceled, the document is deleted), and documents "
            "which have been modified are saved first.  The current folder is either the folder "
            "the current document is in or the folder as assigned by right-clicking on a folder "
            "in the project view and selecting a menu item there.");
        e ("ProjectRenameFolder", "", &ProjectRenameFolder,
            "Rename the current folder.",
            "Show a dialog allowing the user to rename the current folder.  The current folder "
            "is either the folder of the current document or that assigned by right-clicking on "
            "a folder in the project view and selecting a menu item there.");
        e ("ProjectStats", ".", &ProjectStats,
            "Show some project statistics.",
            "Show a dialog of some project statistics, such as byte/word/line count.");
        e ("ProjectCompile", "", &ProjectCompile,
            "Compile the project.",
            "Run the project's compilation command.  If there is none, this will bring up a dialog "
            "allowing the user to set it up.  If the user cancels this dialog, then no compilation is "
            "done.");
        e ("ProjectCompileSettings", ".", &ProjectCompileSettings,
            "Setup project compilation.",
            "Shows a dialog allowing the user to setup project compilation.");
        e ("ProjectRun", "", &ProjectRun,
            "Run the project.",
            "Execute the project's run tool.  If there is none, this will bring up a dialog allowing "
            "the user to set it up.  If the user cancels this dialog, then no execution is done.");
        e ("ProjectRunSettings", ".", &ProjectRunSettings,
            "Setup project run.",
            "Shows a dialog allowing the user to setup the project run tool.");
    }
    
    void ProjectCompileSettings (View view)
    {
        view.compileCommand.settings (view, "Project Compilation Tool", false);
    }
    
    void ProjectCompile (View view)
    {
        if (view.compileCommand.command === null)
            view.run ("ProjectCompileSettings");
        else
            view.compileCommand.execute (view);
    }
    
    void ProjectRunSettings (View view)
    {
        view.runCommand.settings (view, "Project Run Tool", false);
    }
    
    void ProjectRun (View view)
    {
        if (view.runCommand.command === null)
            view.run ("ProjectRunSettings");
        else
            view.runCommand.execute (view);
    }

    char [] plural (int count, char [] singular, char [] plural)
    {
        static char [256] buffer;
        int length;

        length = std.c.stdio.sprintf (buffer, "%d\t%.*s.\n", count, count == 1 ? singular : plural);
        return buffer [0 .. length];
    }

    void ProjectStats (View view)
    {
        char [4096] buffer;
        char [] t;
        int length;
        int count;
        int [256] incidence;
        int m;

        with (view)
        {
            count = 0;
            for (int c; c < documents.length; c ++)
            {
                Document doc = documents [c];

                for (int d; d < doc.lines.length; d ++)
                {
                    count += doc.lines [d].length;
                    if (d != doc.lines.length - 1)
                        count ++;
                }
            }
            t ~= plural (count, "byte", "bytes");

            count = 0;
            for (int c; c < documents.length; c ++)
            {
                Document doc = documents [c];

                for (int li; li < doc.lines.length; li ++)
                {
                    char [] l = doc.lines [li];
                    bit o = false;

                    for (int d; d < l.length; d ++)
                    {
                        incidence [std.ctype.tolower (l [d])] ++;
                        if (std.ctype.isalnum (l [d]))
                        {
                            if (o)
                                count ++;
                            o = !o;
                        }
                    }
                }
            }
            t ~= plural (count, "word", "words");

            count = 0;
            for (int c; c < documents.length; c ++)
                count += documents [c].lines.length;
            t ~= plural (count, "line", "lines");

            t ~= plural (documents.length, "document", "documents");
            t ~= "\n";
             
            char [26] most;
            char [] mostly = most;

            for (int d = 0; d < most.length; d ++)
            {
                m = 0;
                for (int c = 0; c < 256; c ++)
                    if (std.ctype.isalpha (c) && (!m || incidence [c] > incidence [m]))
                        m = c;
                if (incidence [m] == 0)
                {
                    mostly = most [0 .. d];
                    break;
                }
                incidence [m] = 0;
                most [d] = m;
            }

            if (mostly.length)
            {
                length = std.c.stdio.sprintf (buffer, "'%.*s'\n\tcharacter incidence order.\n", mostly);
                t ~= buffer [0 .. length];
            }

            messageBox ("Project Statistics", t);
        }
    }

    void ProjectNewFolder (View view)
    {
        new CreateFolder (view);
    }

    void ProjectDeleteFolder (View view)
    {
        ProjectView.Folder folder = view.currentFolder ();

        folder.remove ();
        global.projectView.paint ();
    }

    void ProjectRenameFolder (View view)
    {
        new RenameFolder (view, view.currentFolder ());
    }

    void ProjectNew (View view)
    {
        with (view)
        {
            projectName = null;
            document = null;
            documents = null;
            ordered = null;
            global.projectView.restart ();
            runCommand = new ExternalTool;
            compileCommand = new ExternalTool;
            newDocument ();
        }
    }

    void saveFolder (View view, ProjectView.Folder folder, OutBuffer o)
    {
        o.write ((uint) folder.folderCount ());
        o.write ((uint) folder.documentCount ());
        
        folder.folderIterate (delegate void (ProjectView.Folder sub)
        {
            o.write ((uint) sub.name ().length);
            o.write (sub.name ());
            o.write ((ubyte) sub.open ());
            saveFolder (view, sub, o);
        });

        folder.documentIterate (delegate void (ProjectView.DocumentRow sub)
        {
            Document doc = sub.base;

            for (int d = 0; ; d ++)
            {
                assert (d < view.documents.length);
                if (doc === view.documents [d])
                {
                    o.write ((uint) d);
                    break;
                }
            }
        });
    }
    
    void write (OutBuffer o, char [] string)
    {
        o.write ((uint) string.length);
        o.write (string);
    }
    
    void write (OutBuffer o, ExternalTool t)
    {
        write (o, t.title);
        write (o, t.command);
        write (o, t.initialDirectory);
        o.write ((ubyte) t.useOutputWindow);
        o.write ((ubyte) t.promptForArguments);
        o.write ((ubyte) t.closeOnExit);
        o.write ((ubyte) t.saveAllFiles);
        o.write ((ubyte) t.deditSelfDevelopment);
    }

    void ProjectSave (View view)
    {
        with (view)
        {
            if (projectName === null)
            {
                char [] [] result;

                with (new FileSelector (true))
                {
                    multiSelect = false;
                    addFilter ("Dedit Project", "*.dprj");
                    addFilter ("All Files", "*");
    
                    result = run ();
                    if (result === null)
                        return;
                }

                projectName = result [0];
            }

            OutBuffer o = new OutBuffer ();

            o.write ("Dedit Project File\n");
            o.write ((ubyte) 7); // Revision.
            
            write (o, compileCommand); /* Revision 4 */
            write (o, runCommand); /* Revision 6 */
           
            /* Revision 3: Macro. */ 
            o.write ((uint) macro.length);
            for (int c; c < macro.length; c ++)
            {
                o.write ((uint) macro [c].length);
                o.write (macro [c]);
            }                
            
            /* Documents. */
            o.write ((uint) documents.length);

            for (int c; c < documents.length; c ++)
                if (documents [c] === document)
                {
                    o.write ((uint) c);
                    break;
                }

            for (int c; c < documents.length; c ++)
            {
                Document d = documents [c];
                char [] hilite = d.hilite.name ();

                if (d.filename === null)
                {
                    o.write ((uint) d.index);
                    o.write ((uint) d.lines.length);
                    o.write ((uint) hilite.length);
                    o.write (hilite);
                    for (int e; e < d.lines.length; e ++)
                    {
                        o.write ((uint) d.lines [e].length);
                        o.write (d.lines [e]);
                    }
                    d.setModified (false);
                }
                else
                {
                    char [] file = relativeFilename (d.filename, getDirName (projectName));

                    o.write ((uint) 0);
                    o.write ((uint) file.length);
                    o.write (file);
                    o.write ((uint) hilite.length);
                    o.write (hilite);
                }

                o.write ((int) d.line);
                o.write ((int) d.offset);
                o.write ((int) d.selStartLine);
                o.write ((int) d.selStartOffset);
                o.write ((int) d.selEndLine);
                o.write ((int) d.selEndOffset);
                o.write ((int) d.lineOffset);
            }

            saveFolder (view, global.projectView.root, o);

            std.file.write (projectName, (byte []) o.toBytes ());
        }
    }

    void ProjectOpen (View view)
    {
        with (view)
        {
            char [] [] result;

            with (new FileSelector (false))
            {
                multiSelect = false;
                addFilter ("Dedit Project", "*.dprj");
                addFilter ("All Files", "*");
                result = run ();
                if (result === null)
                    return;
            }

            if (!view.confirmModified ())
                return;

            if (projectName)
                run ("ProjectSave");
            else
            {
                for (int c; c < documents.length; c ++)
                    if (documents [c].filename === null && documents [c].modified)
                    {
                        char [] r = messageBox ("Throw away files?", "Some unnamed files will be thrown away.\nAre you sure you want to discard them?", MB.YesNo);

                        if (r == "No")
                            return;
                        break;
                    }
            }

            projectName = result [0];

            run ("ProjectReload");
        }
    }

    ubyte [] data;

    ubyte [] readUBytes (int length)
    {
        ubyte [] result;

        if (data.length < length)
            throw new Error ("Couldn't read any more bytes");
        result = data [0 .. length];
        data = data [length .. data.length];
        return result;
    }

    char [] readChars (int length)
    {
        return (char []) readUBytes (length);
    }

    ubyte readUByte ()
    {
        return readUBytes (1) [0];
    }

    uint readUInt ()
    {
        ubyte [] data = readUBytes (4);

        return data [0] | (data [1] << 8) | (data [2] << 16) | (data [3] << 24);
    }

    char [] readString ()
    {
        uint length = readUInt ();

        return readChars (length);
    }

    int readInt ()
    {
        return readUInt ();
    }
    
    int revision;

    ExternalTool loadExternalTool ()
    {
        ExternalTool t = new ExternalTool;
       
        t.title = readString ();
        t.command = readString ();
        if (revision < 7)
            readString ();
        t.initialDirectory = readString ();
        t.useOutputWindow = (bit) readUByte (); 
        t.promptForArguments = (bit) readUByte ();
        t.closeOnExit = (bit) readUByte ();
        if (revision >= 5)
        {
            t.saveAllFiles = (bit) readUByte ();
            t.deditSelfDevelopment = (bit) readUByte ();
        }
        return t;
    }
    
    void loadFolder (View view, ProjectView.Folder folder, Document [] documents)
    {
        uint folderCount = readUInt ();
        uint documentCount = readUInt ();

        for (int c; c < folderCount; c ++)
        {
            ProjectView.Folder sub;

            sub = new ProjectView.Folder (readString ().dup);
            folder.add (sub);
            sub.open ((bit) readUByte ());
            loadFolder (view, sub, documents);
        }

        for (int c; c < documentCount; c ++)
        {
            int x = readUInt ();

            if (documents [x] !== null)
            {
                global.documentRemoved (documents [x]);
                folder.add (new ProjectView.DocumentRow (documents [x]));
            }
        }
    }

    void ProjectReload (View view)
    {
        char [] [] errors;

        with (view)
        {
            document = null;
            documents = null;
            ordered = null;
            global.projectView.empty ();

            global.projectView.root = new ProjectView.Folder ("Files");

            global.projectView.add (global.projectView.root);

            try data = (ubyte []) std.file.read (projectName);
            catch (Object o)
            {
                messageBox ("Open Problem", "Couldn't open project file '" ~ projectName ~ "'.");
                newDocument ();
                projectName = null;
                return;
            }

            char [] mark = readChars (19);

            if (mark != "Dedit Project File\n")
            {
                messageBox ("Open Problem", "This is not a Dedit project file");
                return;
            }

            revision = readUByte ();

            if (revision < 0 || revision > 7)
            {
                messageBox ("Open Problem", "Unknown dedit project file revision number");
                return;
            }
            
            if (revision >= 4)
                compileCommand = loadExternalTool ();
            else
                compileCommand = new ExternalTool;                
                
            if (revision >= 6)
                runCommand = loadExternalTool ();
            else
                runCommand = new ExternalTool;

            if (revision >= 3)
            {
                macro.length = readUInt ();
                for (int c; c < macro.length; c ++)
                    macro [c] = readString ();
            }
            else
                macro.length = 0;    

            int count = readUInt ();
            int current = 0;
            Document [] list;

            if (revision >= 1)
                current = readUInt ();

            global.projectView.deferPaint = true;

            for (int c; c < count; c ++)
            {
                int index = readUInt ();

                if (index)
                {
                    newDocument ();
                    document.index = index;

                    int linesLength = readUInt ();
                    char [] hilite = readString ();

                    document.syntaxHighlighter (hilite);
                    document.lines.length = linesLength;
                    document.highs.length = linesLength;

                    for (int e; e < linesLength; e ++)
                        document.lines [e] = readString ();
                }
                else
                {
                    char [] filename = readString ();
                    char [] absolute = absoluteFilename (filename, getDirName (projectName));

                    try
                    {
                        std.file.getSize (absolute);
                        loadFile (absolute);
                    }
                    catch (std.file.FileException)
                    {
                        readString ();
                        readInt ();
                        readInt ();
                        readInt ();
                        readInt ();
                        readInt ();
                        readInt ();
                        readInt ();

                        errors ~= filename;
                        if (current >= c)
                            current --;
                        list ~= null;
                        continue;
                    }

                    char [] hilite = readString ();

                    document.syntaxHighlighter (hilite);
                }

                Document d = document;

                list ~= d;
                d.line = readInt ();
                d.offset = readInt ();
                d.selStartLine = readInt ();
                d.selStartOffset = readInt ();
                d.selEndLine = readInt ();
                d.selEndOffset = readInt ();
                d.lineOffset = readInt ();
            }

            if (revision >= 2)
                loadFolder (view, global.projectView.root, list);
            else for (int c; c < list.length; c ++)
                global.projectView.root.add (new ProjectView.DocumentRow (list [c]));

            if (errors.length)
            {
                char [] message;

                message = "These files cannot be opened and will be removed from the project:\n\n";
                for (int c; c < errors.length; c ++)
                    message ~= errors [c] ~ "\n";

                messageBox ("Problems Reading Files", message, MB.OK);
            }

            if (current >= 0)
                document = documents [current];
            else
                newDocument ();

            paint ();
            global.projectView.deferPaint = false;
            global.view.setCaption ();
            global.projectView.root.open (true);
        }
    }
}

